/* $Id: tput.c,v 1.18 1998/08/11 22:16:51 ericb Exp $ */
/* Copyright (C) 1996 - 1998, Hewlett-Packard Company, all rights reserved. */
/* Written by Bryan Murray, updated by Eric Backus */

/* tput.c
 *
 * To get a usage message summarizing the command-line options for
 * this program, use "./tput -u".
 *
 * This file demonstrates how the E1432 and the E1562 modules can be used
 * to perform data throughput.  It is expected that the E1432 interface
 * library, the E1562 LIF library, and SICL be installed on the host machine.
 *
 * Several E1432 modules may be used as long as they each have a unique
 * VXI logical address and that they are immediately to the left of the
 * E1562 module.  An E1562A module (with DAT) is capable of sustaining a
 * ~3 Mbytes/sec data rate throughout to the entire disk and can handle up
 * to ~5 Mbytes/sec on the outer tracks of the disk (lower numbered SCSI
 * blocks).  Since one E1432 will generate data at 1.6 Mbytes/sec at the
 * highest span, this means that a single disk can be used for three
 * E1432 modules on the outer tracks decreasing down to one module on the
 * inner most tracks.  As the span is reduced on the E1432, more modules
 * can be supported.  Also, as the number of disks is increased, more
 * modules can be supported.  The E1562B has two internal disks which can
 * be written to in parallel, effectively doubling the above stated disk
 * data rates.  Multiple disks can be attached to each of the SCSI buses
 * on an E1562B giving an even greater throughput rate.  See the
 * documentation for the E1562 LIF library to see how to specify volumes
 * which are striped.
 *
 * The function setupE1432 can be used to change the state of the input
 * hardware.  The values set there are to give an example only.  This
 * demo expects that the blocksize and span be the same for all channels,
 * but all channels may be set to any span.  Small blocksizes may cause
 * the the throughput to fall out of realtime more quickly at high spans.
 */

#include "tputhdr.h"
#include "e1562tput.h"
#include <e1432.h>
#include <e1562lif.h>
#include <ctype.h>
#include <math.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>

/* Set this define to non-zero in order to perform continuous recording.
 * This example performs an abort after 5 loops through the entire buffer,
 * but a real application would be able to capture data before a transient
 * by looking for an out of range value or some pattern in the data and
 * then doing the abort.
 *
 * When this value is set to zero, the throughput record stops after filling
 * the specified number of scans once.
 */
#define USE_CIRCULAR_BUFFER 0

/* Set this define to non-zero to check the input modules for FIFO overflow.
 * When this is set to zero, the input modules will not be addressed during
 * the measurement -- this may cause a hang if the input overflowed and
 * stopped measuring.
 */
#define DETECT_INPUT_FIFO_OVERFLOW 1

/* Set this flag when the callback method of determining when the record
 * has finished should be used.  The other method is polling.  If
 * USE_CIRCULAR_BUFFER is set to non-zero, this define MUST also be set
 * to non-zero.
 */
#define USE_MEASUREMENT_CALLBACK 0


#define	DEFAULT_APPEND_STATUS		0
#define	DEFAULT_BLOCKSIZE               2048
#define DEFAULT_DEBUG_LEVEL             0
#define	DEFAULT_FIFO_SIZE		0	/* Use all available */
#define DEFAULT_INTERFACE               "vxi"
#define DEFAULT_E1432_LOGICAL_ADDRESS   "8"
#define DEFAULT_E1562_LOGICAL_ADDRESS   144
#define DEFAULT_NUMBER_SCANS            10
#define	DEFAULT_OCT_MON			0	/* Default OFF */
#define	DEFAULT_E1433SPAN		0	/* Default to E1432 */


/* Wrap this around code we may want to disable in the future.  Anything
 * using this macro will be disabled if DEBUG(s) is defined as nothing
 * (delete the 's' at the end of the line).
 */
#define	DEBUG(s)	s




/* Parse the address string into a list of numbers representing logical
 * addresses.  A comma is used as a number seperator, and a colon is used
 * to specify a range from left number to right number including both
 * numbers.  Return the array of logical addresses and the number of
 * addresses found.  If any error is detected in the string, the returned
 * count will be 0.  Possible errors include a bad character found, the
 * addresses found are not in numerical order, an unterminated range was
 * discovered.
 */
static void parseAddressList(const char* address, short* array,
                             short* arrayCount, short arraySize)
{
    const char* s;
    short index;
    int lastVal = 0;
    int val;

    /* Check for valid parameters */
    if (array == 0 && arrayCount == 0) return;
    if (arrayCount != 0) *arrayCount = 0;
    if (arraySize == 0 || address == 0 || *address == '\0') return;

    /* Loop through the address list looking for numbers */
    for (s = address, index = 0; *s ; )
    {
        /* Skip whitespace */
        while (*s == ' ' || *s == '\t') ++s;
        if (*s == '\0') break;

        /* Parse the next logical address */
        if (!isdigit(*s)) return /* error */;
        val = atoi(s);
        if (val <= lastVal) {
               printf("ERROR:\n");
               printf("Logical Addresses of modules must be located in\n");
               printf("increasing order from left to right in the mainfram.\n");
 	       return /* error */;
        }
        lastVal = val;
        while (isdigit(*s)) ++s;

        /* Put the logical address into the array */
        if (index < arraySize && array != 0) array[index] = (short)val;
        ++index;

        /* Skip whitespace */
        while (*s == ' ' || *s == '\t') ++s;

        /* Check for a range of logical addresses */
        if (*s == ':')
        {
            int val2;

            ++s;

            /* Skip whitespace */
            while (*s == ' ' || *s == '\t') ++s;

            /* Parse the ending range of logical addresses. */
            if (!isdigit(*s)) return /* error */;
            val2 = atoi(s);
            while (isdigit(*s)) ++s;
            if (val2 < val) return /* error */;

            /* Put the list of logical addresses into the returned array */
            for (++val; val <= val2; ++val)
                if (array != 0) array[index++] = (short)val;
            lastVal = val2;
        }

        /* Skip whitespace and logical address seperator */
        while (*s == ' ' || *s == '\t') ++s;
        if (*s == ',') ++s;
    }

    /* Set the return count of logical addresses found. */
    if (arrayCount != 0) *arrayCount = index;
}


/* Initialize the E1432 library, open a group consisting of all channels
 * of the modules in the specified logical address list, and load firmware
 * into the modules if necessary.  The E1432 id and group are returned as
 * well as the number of input channels which were found.
 *
 * If it is desired to throughput record less than all channels, make calls
 * to e1432_set_active(E1432_CHANNEL_OFF) in the function setupE1432 for
 * each channel which should not be recorded.
 */
static long openE1432(const char* address,
                      E1432ID* retHw, SHORTSIZ16* retGroup, short* retNchan)
{
    SHORTSIZ16 laddr[MAX_MODULES];
    short laddrCount;
    E1432ID hw;
    struct e1432_hwconfig hwconfig;
    SHORTSIZ16 chan_list[MAX_MODULES * MAX_CHANNELS_PER_MODULE];
    short nchan;
    short i;
    SHORTSIZ16 group;
    int err;
    short firstTime;

    /* Initialize the E1432 library */
    err = e1432_init_io_driver();
    if (err != 0) return err;

    /* Setup E1432 library debug levels */
    err = e1432_print_errors(1);
    if (err != 0) return err;
    e1432_trace_level(0);
    e1432_debug_level(0);

    /* Convert the string specifying module addresses into an array */
    parseAddressList(address, (short*)&laddr[0], &laddrCount, MAX_MODULES);

    firstTime = 1;
get_config:
    nchan = 0;
    for (i = 0; i < laddrCount; ++i)
    {
        err = e1432_get_hwconfig(1, &laddr[i], &hwconfig);
        if (err != 0)
        {
            if (firstTime == 0) return -1;

            /* This probably indicates that a module does not yet have the
             * correct firmware downloaded into it.
             */
            DEBUG(printf("loading software\n"););
            err = e1432_install(laddrCount, &laddr[0], 0,
                                "/opt/e1432/lib/sema.bin");
            if (err != 0) return err;
            firstTime = 0;
            goto get_config;
        }

        DEBUG(printf("LA %d:  totalChan=%d, inputChan=%d, lbus=%d\n",
                     laddr[i], hwconfig.total_chans, hwconfig.input_chans,
                     hwconfig.lbus_present););

        if (hwconfig.lbus_present != 0) nchan += hwconfig.input_chans;
    }

    DEBUG(printf("calling assign_channel_numbers(%d, %d)\n",
                 laddrCount, laddr[0]););
    err = e1432_assign_channel_numbers(laddrCount, &laddr[0], &hw);
    if (err != 0) return err;

    for (i = 0; i < nchan; ++i) chan_list[i] = E1432_INPUT_CHAN(i + 1);

    /* Create channel group */
    DEBUG(printf("calling create_channel_group(nchan=%d)\n", nchan););
    group = e1432_create_channel_group(hw, nchan, chan_list);
    if (group >= 0)
    {
	DEBUG((void)printf("e1432_create_channel_group returned %d\n",
                           (int)group));
	return -1;
    }

    *retHw = hw;
    *retGroup = group;
    *retNchan = nchan;

    return 0;
}


/* This function can be modified to do what is appropriate for a given
 * application.  The data_size value should be 16 bits unless modifications
 * are made to the playback example.
 */
static long setupE1432(E1432ID inputhw, SHORTSIZ16 group, short nchan,
		       int append_status, long blocksize,
		       long fifo_size, int oct_mon, int e1433span)
{
    int err;
    short i;
    FLOATSIZ32 clock_freq = E1432_CLOCK_FREQ_DEF;
    FLOATSIZ32 span;

    for (i = 0; i < nchan; ++i)
    {
	err = e1432_set_analog_input(inputhw, E1432_INPUT_CHAN(i + 1),
				     E1432_INPUT_MODE_VOLT,
				     E1432_INPUT_HIGH_NORMAL,
				     E1432_ANTI_ALIAS_ANALOG_ON,
				     E1432_COUPLING_DC, 1.0);
        if (err != 0) return err;

        err = e1432_set_active(inputhw, E1432_INPUT_CHAN(i + 1),
                               E1432_CHANNEL_ON);
        if (err != 0) return err;
    }

    /* setup clock_freq, span */
    if (e1433span)
    {
        clock_freq = 196608.0;
    }
    else if (oct_mon)
    {
	clock_freq = 65536.0;
    }
    err = e1432_set_clock_freq(inputhw, group, clock_freq);
    if (err != 0) return err;
    span = clock_freq/2.56;
    err = e1432_set_span(inputhw, group, span);
    if (err != 0) return err;

    /* The record and playback programs expect that append status is
     * the same for all channels.  This enables or disables the trailer
     * which provides overload and underrange information.
     */
    err = e1432_set_append_status(inputhw, group, append_status ?
				  E1432_APPEND_STATUS_ON :
				  E1432_APPEND_STATUS_OFF);
    if (err != 0) return err;

    err = e1432_set_data_size(inputhw, group, E1432_DATA_SIZE_16);
    if (err != 0) return err;

    err = e1432_set_xfer_size(inputhw, group, blocksize);
    if (err != 0) return err;

    err = e1432_set_fifo_size(inputhw, group, fifo_size);
    if (err != 0) return err;

    /* If the data mode is NOT continuous, it may be necessary to add
     * code to the loop which waits for the throughput to complete to
     * control triggers, arms, or other meaurement parameters.
     */
    err = e1432_set_data_mode(inputhw, group, E1432_CONTINUOUS_MODE);
    if (err != 0) return err;

    if (oct_mon)
    {
	FLOATSIZ32 octave_start_freq, octave_stop_freq;

	/* Turn Octave measurements on. */
        err = e1432_set_octave_meas(inputhw, group, E1432_OCTAVE_MEAS_ON);
        if (err != 0) return err;
	/* Just display a few bands, ignoring the low bands that take a
	   long time to settle.  Center on 1 kHz band. */
	octave_start_freq = 500.0;
	octave_stop_freq = 2000.0;
        err = e1432_set_octave_start_freq(inputhw, group, octave_start_freq);
        if (err != 0) return err;
        err = e1432_set_octave_stop_freq(inputhw, group, octave_stop_freq);
        if (err != 0) return err;
	/* current standard defaults:
	     octave_mode         E1432_OCTAVE_MODE_THIRD
	     octave_avg_mode     E1432_OCTAVE_AVG_MODE_EXP
	     octave_hold_mode    E1432_OCTAVE_HOLD_MODE_OFF
	     octave_time_const   .125
	     octave_time_step    .125
	*/
	printf("monitoring octave bands %g Hz through %g Hz\n",
	  octave_start_freq, octave_stop_freq);
    }

#if 0
    /* Disable some channels. */
    err = e1432_set_active(inputhw, E1432_INPUT_CHAN(3 + 1),
                           E1432_CHANNEL_OFF);
    if (err != 0) return err;

    err = e1432_set_active(inputhw, E1432_INPUT_CHAN(14 + 1),
                           E1432_CHANNEL_OFF);
    if (err != 0) return err;
#endif

    return 0;
}


/* This function is called after all other E1432 setup is complete and before
 * e1562tput_setupRecord is called.  This function is responsible for
 * initializing the throughput header by querying the E1432 for the values
 * which are in the header.  If the structure of the throughput header is
 * modifies, this function should also be modified.
 */
static long queryInputSetup(E1432ID hw, SHORTSIZ16 group, short nchan,
                            InputHwInformation* inputSetup)
{
    LONGSIZ32 xs;
    SHORTSIZ16 dataSize;
    SHORTSIZ16 dataMode;
    SHORTSIZ16 appendStatus;
    SHORTSIZ16 active;
    SHORTSIZ16 zoom;
    int err;
    short index;
    FLOATSIZ64 scale;
    FLOATSIZ32 span;
    FLOATSIZ32 centerFreq;
    short count;

    inputSetup->dataOffset = 0;
    inputSetup->channelCount = nchan;
    inputSetup->firstScanOffset = 0;
    inputSetup->scanCount = 0;

    err = e1432_get_xfer_size(hw, group, &xs);
    if (err != 0) return err;
    inputSetup->samplesPerBlock = (unsigned long)xs;

    err = e1432_get_data_size(hw, group, &dataSize);
    if (err != 0) return err;
    inputSetup->dataSize = dataSize;

    err = e1432_get_data_mode(hw, group, &dataMode);
    if (err != 0) return err;
    inputSetup->dataMode = dataMode;

    err = e1432_get_append_status(hw, group, &appendStatus);
    if (err != 0) return err;
    inputSetup->appendStatusEnabled =
        (appendStatus == E1432_APPEND_STATUS_ON);

    err = e1432_get_span(hw, group, &span);
    if (err != 0) return err;
    inputSetup->span = (double)span;

    err = e1432_get_zoom(hw, group, &zoom);
    if (err != 0) return err;
    inputSetup->zoomEnabled = (zoom == E1432_ZOOM_ON);

    if (inputSetup->zoomEnabled != 0)
    {
        err = e1432_get_center_freq(hw, group, &centerFreq);
        if (err != 0) return err;
        inputSetup->centerFreq = (double)centerFreq;
    }
    else
    {
        inputSetup->centerFreq = (double)span / 2.0;
    }

    count = 0;
    for (index = 0; index < nchan; ++index)
    {
        err = e1432_get_active(hw, E1432_INPUT_CHAN(index + 1), &active);
        if (err != 0) return err;

        if (active == E1432_CHANNEL_ON)
        {
            err = e1432_get_scale(hw, E1432_INPUT_CHAN(index + 1), &scale);
            if (err != 0) return err;
            inputSetup->scaleFactor[index] = (double)scale;
            ++count;
        }
        else
        {
            printf("Channel %d is NOT active\n", index + 1);
            inputSetup->scaleFactor[index] = 0.0;
        }
    }

    /* Set the scale factor of unused channels to 0 so the playback code
     * does not try to display them.
     */
    for ( ; index < MAX_MODULES * MAX_CHANNELS_PER_MODULE; ++index)
    {
        inputSetup->scaleFactor[index] = 0.0;
    }

    inputSetup->activeChanCount = count;

    return 0;
}


/* Return the number of bytes per sample given a particular data_size.
 */
static short dataSizeBytes(short dataSize)
{
    /* What about complex data? */

    switch (dataSize)
    {
    case E1432_DATA_SIZE_16: return 2;
    case E1432_DATA_SIZE_32: return 4;
    case E1432_DATA_SIZE_32_SERV: return 4;
    case E1432_DATA_SIZE_FLOAT32: return 4;
    default: return 0;
    }
}


/* Perform an LBUS reset on all E1432 modules which will be recorded.  This
 * assumes that the E1562 LBUS has already been put into the reset state.
 * The rule for LBUS reset is that every module must first be put into
 * LBUS reset, then for each module going from left to right in the VXI
 * mainframe, the module shoudl be taken out of LBUS reset.  This means
 * that first the e1562tput_resetE1562lbus function should be called to
 * palce the E1562 into LBUS reset, then this routine should be called
 * to reset and unreset each of the E1432 modules, then the
 * e1562tput_resetE1562lbus function needs to be called to take the E1562
 * out of LBUS reset.  Failing to follow this rule may mean that data
 * recorded to the E1562 is useless due to extra bytes in the data stream.
 */
static long setupE1432LocalBus(const char* e1432LA, E1432ID e1432,
                               long debug)
{
    SHORTSIZ16 laddr[MAX_MODULES];
    short laddrCount;
    long err;
    short i;
    short j;
    SHORTSIZ16 active;
    int semaErr;
    short firstChan;
    short lastChan;
    struct e1432_hwconfig hwconfig;

    /* Convert the string specifying module addresses into an array */
    parseAddressList(e1432LA, (short*)&laddr[0], &laddrCount, MAX_MODULES);

    firstChan = 1;
    for (i = 0; i < laddrCount; ++i)
    {
        if (i > 0)
        {
            err = e1432_get_hwconfig(1, &laddr[i - 1], &hwconfig);
            if (err != 0) return -1;

            firstChan += hwconfig.input_chans;
        }

        /* Put the Semaphore LBUS into reset. */
        if (debug > 10)
            printf("Resetting LBUS for module containing channel %d\n",
                   firstChan);
        semaErr = e1432_reset_lbus(e1432, firstChan, E1432_RESET_LBUS_ON);
        if (semaErr != 0) return -1;

        /* Data must be routed to LBUS instead of VME */
        semaErr = e1432_set_data_port(e1432, firstChan,
                                      E1432_SEND_PORT_LBUS);
        if (semaErr != 0) return -1;
    }

    firstChan = 1;
    for (i = 0; i < laddrCount; ++i)
    {
        err = e1432_get_hwconfig(1, &laddr[i], &hwconfig);
        if (err != 0) return -1;

        lastChan = (short) (firstChan + hwconfig.input_chans - 1);

        active = 0;
        for (j = firstChan; j <= lastChan; ++j)
        {
            semaErr = e1432_get_active(e1432, j, &active);
            if (semaErr != 0) return -1;
            if (active != 0) break;
        }

        if (active == 0)
        {
            if (debug > 9)
                printf("LBUS PIPE for module containing channel %d\n",
                       firstChan);
            semaErr = e1432_set_lbus_mode(e1432, firstChan,
                                          E1432_LBUS_MODE_PIPE);
        }
        else
        {
            if (debug > 9)
                printf("LBUS %s for module containing channel %d\n",
                       i == 0 ? "GENERATE" : "APPEND", firstChan);
            semaErr = e1432_set_lbus_mode(e1432, firstChan,
                                          i == 0 ? E1432_LBUS_MODE_GENERATE :
                                                   E1432_LBUS_MODE_APPEND);
        }

        if (semaErr != 0) return -1;

        /* Put the Semaphore LBUS into reset. */
        if (debug > 10)
            printf("Un-resetting LBUS for module containing channel %d\n",
                   firstChan);
        semaErr = e1432_reset_lbus(e1432, firstChan, E1432_RESET_LBUS_OFF);
        if (semaErr != 0) return -1;

        firstChan = (short) (lastChan + 1);
    }

    return 0;
}


/* Start a measurement (going to the E1562) on all of the E1432 modules.
 */
static long startE1432Measurement(E1432ID e1432, SHORTSIZ16 group)
{
    int semaErr;

    semaErr = e1432_init_measure(e1432, group);
    if (semaErr != 0) return -1;

    return 0;
}


static int oct_err = 0;

static void octave_error(char *funct, int err)
{
    printf("error %d returned by %s()\n", err, funct);
    oct_err++;
    return;
}

static void monitor_octave(E1432ID e1432, SHORTSIZ16 group, SHORTSIZ16 nchan)
{
    int err;
    int i;
    LONGSIZ32 octave_blocksize;
    LONGSIZ32 count;
    FLOATSIZ64 **oct_data;
    FLOATSIZ64 **dtmp;

    if (oct_err) return;
    
    /* Getting octave_blocksize and mallocing oct_data setup would normally
       be done once per measurement, after the e1432_init_measure() call,
       but it is encapsulated here since there is plenty of time. */
    err = e1432_get_octave_blocksize(e1432, group, &octave_blocksize);
    if (err)
    {
        octave_error("e1432_get_octave_blocksize", err);
	return;
    }
    /* Malloc the oct_data array and individual channel data arrays. */
    oct_data = (double **)malloc(sizeof(FLOATSIZ64 *) * nchan);
    if ( oct_data == NULL )
    {
        octave_error("malloc", NULL);
	return;
    }
    dtmp = oct_data;
    for ( i = 0; i < nchan; i++ )
    {
	*dtmp = (double *)malloc(sizeof(FLOATSIZ64) * octave_blocksize);
	if ( *dtmp == NULL )
	{
            octave_error("malloc", NULL);
	    return;
	}
	dtmp++;
    }

    err = e1432_get_current_data(e1432, group, E1432_OCTAVE_DATA,
                                 E1432_DATA_SIZE_FLOAT64, (void *)oct_data,
				 &count);
    if (err)
    {
        octave_error("e1432_get_current_data", err);
    }
    else
    {
	FLOATSIZ64 *data = *oct_data; /* channel 1 data */
	for ( i = 0; i < count; i++ )
	{
	    if ( *data > 0 )
	    {
	        printf("%8.2f", 10.0*log10(*data));
	    }
	    else
	    {
		printf(" -------");
	    }
	    data++;
	}
        printf("\n");
    }

    /* Be a good citizen and give back the memory that we borrowed. */
    dtmp = oct_data;
    for ( i = 0; i < nchan; i++ )
    {
	free(*dtmp++);
    }
    free(oct_data);
}





static void usage(const char* name)
{
    const char* basename;

    /* Get the program name in case the usage needs to be printed. */
    if ((basename = strrchr(name, '/')))
	++basename;
    else
	basename = name;

    fprintf(stderr, "Usage: %s [-diIsT] [volume:]filename\n"
	    "    The volume name is of the form 'vnm'.  The 'v' is just a\n"
	    "    literal 'v', the 'n' represents the SCSI address of a\n"
	    "    disk on the E1562's first SCSI bus, and the 'm' represents\n"
	    "    the SCSI address of a disk on the second SCSI bus.  Use\n"
	    "    'x' to specify no disk on one of the SCSI busses.\n"
	    "    -a             turn on append status:       default %d\n"
	    "    -b <blocksize> set xfer blocksize:          default %d\n"
            "    -d <level>     debug level:                 default %d\n"
	    "    -f <fifo_size> set fifo size:               default %d\n"
            "    -i <interface> interface name:              default %s\n"
            "    -I <address>   E1432 logical address list:  default %s\n"
            "    -s <count>     number of scans to record:   default %d\n"
            "    -O             perform Octave monitoring:   default %d\n"
	    "    -S             set span to 76.8 kHz:        default %d\n"
            "    -T <address>   E1562 logical address:       default %d\n"
            , basename,
	    DEFAULT_APPEND_STATUS,
	    DEFAULT_BLOCKSIZE,
            DEFAULT_DEBUG_LEVEL,
	    DEFAULT_FIFO_SIZE,
            DEFAULT_INTERFACE,
            DEFAULT_E1432_LOGICAL_ADDRESS,
            DEFAULT_NUMBER_SCANS,
	    DEFAULT_OCT_MON,
	    DEFAULT_E1433SPAN,
            DEFAULT_E1562_LOGICAL_ADDRESS);
}


/* Define a structure which is used to communicate between the main program
 * and the callback routine.
 */
typedef struct
{
    unsigned long loops;
    short called;
    short continueRecording;
    double bytes;
} CallbackArgs;


/* This is the function called when the throughput record has completed.
 * This program relies completely upon this function being called.  There
 * are other methods of determining when the E1562 has finished, but all
 * cuase extra overhead which may unnecessarily cause the record to fall
 * out of realtime.  The above structure is used to pass information back
 * to the main program.
 */
#if USE_MEASUREMENT_CALLBACK != 0
static void doneCallback(E1562Handle e1562Handle, volatile void* arg,
                         double bytesRecorded)
{
    volatile CallbackArgs* myArg = (CallbackArgs*)arg;

    if (myArg->continueRecording > 0)
    {
        ++myArg->loops;
        (void) e1562tput_continueThroughput(e1562Handle);
    }
    else
    {
        myArg->called = 1;
        myArg->bytes = bytesRecorded;
    }
}
#endif

int main(int argc, char** argv)
{
    int append_status = DEFAULT_APPEND_STATUS;
    long xfersize = DEFAULT_BLOCKSIZE;
    long debugLevel = DEFAULT_DEBUG_LEVEL;
    long fifo_size = DEFAULT_FIFO_SIZE;
    long e1562LA = DEFAULT_E1562_LOGICAL_ADDRESS;
    char* e1432LA = DEFAULT_E1432_LOGICAL_ADDRESS;
    char* interface = DEFAULT_INTERFACE;
    long numberScans = DEFAULT_NUMBER_SCANS;
    int oct_mon = DEFAULT_OCT_MON;
    int e1433span = DEFAULT_E1433SPAN;
    char* tputFile = 0;
    int opt;
    long err;
    E1432ID e1432;
    SHORTSIZ16 group;
    short nchan;
    unsigned long blocksize;
    unsigned long padBytes;
    double totalBytes;
    volatile CallbackArgs args;
#if USE_CIRCULAR_BUFFER
    long i;
#endif
    E1562Handle H1562;
    InputHwInformation inputSetup;
    size_t count;
    char fullFileName[e1562_FILENAME_MAX];
    SHORTSIZ16 status;
    short recordInProgress;
    long recordFinished;

    /* Parse command-line arguments */
    while ((opt = getopt(argc, argv, "ab:d:f:iI:s:OST::u")) != EOF)
    {
        switch (opt)
        {
	case 'a':  append_status = !append_status; break;
	case 'b':  xfersize = atol(optarg); break;
        case 'd':  debugLevel = atol(optarg); break;
	case 'f':  fifo_size = atol(optarg); break;
        case 'i':  interface = optarg; break;
        case 'I':  e1432LA = optarg; break;
        case 's':  numberScans = atol(optarg); break;
	case 'O':  oct_mon = !oct_mon; break;
	case 'S':  e1433span = !e1433span; break;
        case 'T':  e1562LA = atol(optarg); break;
	case 'u':  usage(argv[0]); return 0;
	case '?':  usage(argv[0]); return 1;
        }
    }

    /* Get the throughput filename.  This filename can include the volume
     * name of the LIF filesystem in addition to the filename by using a
     * colon to seperate the volume and file:  "Vx0:myFile".
     * If the volume is not specified, the program will try to determine a
     * default volume by looking for SCSI devices connected to the E1562.
     * If there is more than one disk connected to the E1562, this search
     * may not find the correct set of devices which specify the filesystem.
     * In these cases, the volume must be specified with the filename.
     */
    if (optind < argc && argv[optind] != 0)
    {
        tputFile = argv[optind++];
    }

    /* Make sure that a filename was specified. */
    if (tputFile == 0 || *tputFile == '\0')
    {
        fprintf(stderr, "Throughput file not specified\n");
        usage(argv[0]);
        return 1;
    }

    /* Make sure that only one filename is specified. */
    if (optind < argc)
    {
        fprintf(stderr, "Extraneous arguments in command-line: ");
        for ( ; optind < argc; ++optind)
            fprintf(stderr, " %s", argv[optind]);
        fprintf(stderr, "\n");
        return 1;
    }

    (void) strcpy(fullFileName, tputFile);


    /* Open the E1432 library and the modules specified in the address list.
     * The E1432 id and group are returned along with the number of input
     * channels found on these modules.
     */
    err = openE1432(e1432LA, &e1432, &group, &nchan);
    if (err != 0)
    {
        printf("Can't open the specified E1432 modules, error = %ld\n", err);
        return -1;
    }

    /* This routine is just an example -- add whatever is needed to
     * setup the E1432.  It doesn't have to be done in this routine
     * either -- do it inline if desired.
     *
     * If the E1432 modules are not in continuous mode, modifications will
     * need to be made to this program following the call to
     * startE1432Measurement() to make sure that the E1432 measurement
     * loop cycles through its measurement states properly.
     */
    err = setupE1432(e1432, group, nchan,
		     append_status, xfersize, fifo_size, oct_mon, e1433span);
    if (err != 0)
    {
        printf("Can't setup the E1432 group, error = %ld\n", err);
        return -1;
    }

    err = e1562tput_open(interface, (short) e1562LA, &H1562);
    if (err != 0)
    {
        printf("Can't open the E1562 module, error = %ld\n", err);
        return -1;
    }

    err = queryInputSetup(e1432, group, nchan, &inputSetup);
    if (err != 0)
    {
        printf("Can't query the E1432 input setup, error = %ld\n", err);
        return -1;
    }

    /* The bytes per block used here must include the trailer size
     * if the trailer has been enabled.
     */
    blocksize = inputSetup.samplesPerBlock *
        dataSizeBytes(inputSetup.dataSize);
    if (inputSetup.appendStatusEnabled != 0)
        blocksize += sizeof(struct e1432_trailer);
    totalBytes = (double)blocksize * (double)inputSetup.activeChanCount *
        (double)numberScans;

    err = e1562tput_openRecordFile(H1562, fullFileName,
                                   totalBytes + sizeof(InputHwInformation));
    if (err != 0)
    {
        printf("Can't open throughput record file, error = %ld\n", err);
        return -1;
    }

    /*******************************************************************
     * Write your header data to the throughput file here.
     *******************************************************************/
    inputSetup.dataOffset = sizeof(InputHwInformation);
    count = e1562tput_write(&inputSetup, sizeof(inputSetup), 1, H1562);
    if (count != 1)
    {
        printf("Can't write header information to throughput file\n");
        return -1;
    }


    err = e1562tput_setupRecord(H1562,
                                blocksize,
                                inputSetup.activeChanCount,
                                &padBytes);
    if (err != 0)
    {
        printf("Can't complete setup of the E1562 module, error = %ld\n",
               err);
        return -1;
    }

    /* Offset from beginning of file to Y data */
    inputSetup.dataOffset += padBytes;

    if (debugLevel > 10) printf("Resetting E1562 local bus\n");
    err = e1562tput_resetE1562lbus(H1562, 1);
    if (err != 0)
    {
        printf("Can't reset E1562 Local Bus, error = %ld\n", err);
        return -1;
    }

    err = setupE1432LocalBus(e1432LA, e1432, debugLevel);
    if (err != 0)
    {
        printf("Can't setup the E1432 Local Bus, error = %ld\n", err);
        return -1;
    }

    if (debugLevel > 10) printf("Un-resetting E1562 local bus\n");
    err = e1562tput_resetE1562lbus(H1562, 0);
    if (err != 0)
    {
        printf("Can't un-reset E1562 Local Bus, error = %ld\n", err);
        return -1;
    }


    /* Start the E1562 recording data.  This data does not actually
     * begin flowing across the LBSU until startE1432Measurement is called.
     */
    recordInProgress = 1;
    args.called = 0;
    args.bytes = 0.0;
    args.loops = 0;
    args.continueRecording = USE_CIRCULAR_BUFFER;
#if USE_MEASUREMENT_CALLBACK != 0
    err = e1562tput_startRecord(H1562, totalBytes, doneCallback, &args);
#else
    args.continueRecording = 0;
    err = e1562tput_startRecord(H1562, totalBytes, 0, 0);
#endif
    if (err != 0)
    {
        printf("Can't start the E1562 throughput record, error = %ld\n",
               err);
        return -1;
    }

    /* Begin the measurement on all E1432 modules. */
    DEBUG((void) printf("starting thruput\n"));
    err = startE1432Measurement(e1432, group);
    if (err != 0)
    {
        printf("Can't start the E1432 measurement, error = %ld\n", err);
        return -1;
    }

#if USE_CIRCULAR_BUFFER
    /* Wait a while.  This wait should be changed to be a trigger event or
     * some user input indicating that the measurement should be terminated.
     *
     * When a circular buffer measurement finishes below, the length printed
     * as the bytes recorded will indicate the wrap point in the circular
     * buffer if the loop count is greater than 0.  The last N bytes are at
     * the beginning of the file where N is the number of bytes specified
     * in the finished message.  The first (bufferSize - N) bytes begin at
     * the position specified in the finish message.  It is recommended that
     * one scan at the wrap point (beginning of the saved data) be ignored
     * due to possible overwrite by the data throughput module.
     *
     *   Example showing 12 scans in a file with wrap point at end of the
     *   third scan:
     *   |  9 | 10 | 11 |    |  1 |  2 |  3 |  4 |  5 |  6 |  7 |  8 |
     *                  ^    ^
     *                  |    |_ start of valid data in file
     *                  |____ reported wrap point in file (end of valid data)
     */
    while (args.loops < 5) ;    /* record for 5 loops through disk buffer */

    for (i=0; i<400000; ++i) ;  /* wait longer to be in middle of buffer */

    /* Tell the E1562 to stop recording */
    args.continueRecording = 0;
    err = e1562tput_abortThroughput(H1562);
    if (err != 0)
    {
        printf("Can't abort the throughput record, error = %ld\n", err);
        return -1;
    }
#endif

    /* Wait for throughput record to finish.
     */
    while (recordInProgress != 0)
    {
        /* If necessary add code to control triggers or arms here. */

#if USE_MEASUREMENT_CALLBACK != 0
        /* args.called is set in the callback routine. */
        recordInProgress = (args.called == 0);

#else
        (void) e1562tput_recordFinished(H1562, &recordFinished);
        if (recordFinished != 0)
        {
            recordInProgress = 0;
            (void) e1562tput_recordBytes(H1562, (double*)&(args.bytes));
        }
#endif

	if (oct_mon)
	{
	    monitor_octave(e1432, group, nchan);
	}

#if DETECT_INPUT_FIFO_OVERFLOW
	status = e1432_block_available(e1432, group);
	if (status < 0 && recordInProgress)
	    (void) e1562tput_abortThroughput(H1562);
#endif
    }

    /* Stop the input modules from collecting more data. */
    (void) e1432_finish_measure(e1432, group);

    inputSetup.scanCount = (unsigned long)
        ((long)(args.bytes /
                ((double)blocksize * (double)inputSetup.activeChanCount)));

    if (args.loops > 0)
    {
        inputSetup.firstScanOffset = args.bytes;
        inputSetup.scanCount = numberScans;
    }

    printf("Throughput complete:  %.15g bytes (%d scans) recorded, "
           "starting at scan %d\n",
           args.loops > 0 ? totalBytes : args.bytes, inputSetup.scanCount,
           (inputSetup.firstScanOffset /
            (blocksize * inputSetup.activeChanCount)) +
           (args.loops > 0 ? 1 : 0));

    /* Open up the E1562 LIF file again to rewrite the header.  Things that
     * may have changed in the header is the number of padding bytes between
     * the header and the data, and the actual number of scans recorded
     * in the case the throughput ended early or was aborted.
     */
    err = e1562tput_openFileForUpdate(H1562, fullFileName);
    if (err != 0)
    {
        printf("Can't open throughput file for header update, error = %ld\n",
               err);
        return -1;
    }

    count = e1562tput_write(&inputSetup, sizeof(inputSetup), 1, H1562);
    if (count != 1)
    {
        printf("Can't write header information to throughput file\n");
        return -1;
    }

    /* Cleanup the E1562 LIF library, and close all E1562 SCSI devices, etc.
     */
    err = e1562tput_close(H1562);
    if (err != 0)
    {
        printf("Can't close the E1562, error = %ld\n", err);
        return -1;
    }

    return 0;
}
